% finaquant Financial Analytics - www.finaquant.com
% Copyright: Tunc Ali Ktkcoglu  2012, version: 28Mar2012
% Function for Simple Performance Fee Calculation
% Related web page:
% http://finaquant.com/simple-performance-fee-calculation-for-investment-funds/760

function [Outperform,FundReturn,ExcessReturn,RelativeValues,rHWM,FeeBeforeCF,PerfFeePs] = SimplePerformanceFee(NAVps,BenchmarkReturns,applyHWM,FeeRate,HurdleRate,FloorPs,CapPs)
%     NAVps,              % vector of NAV per share values, length = N+1
%     BenchmarkReturns,   % vector of benchmark returns, length = N
%     applyHWM,           % true or false, if true apply rHWM condition in calculation
%     FeeRate,            % performance fee rate, f.e. 5 for 5%
%     HurdleRate,         % hurdle rate, f.e. 5 for 5%
%     FloorPs,            % performance fee floor (lower limit) per share
%     CapPs               % performance fee cap (higher limit) per share

% check vector sizes
% check if empty vector
if isempty(NAVps) || isempty(BenchmarkReturns)
    error 'Vectors cannot be empty'
end
% check the size of vector
[rows,cols] = size(NAVps);
if min(rows,cols) ~= 1
    error 'NAVps must be a 1xN or Nx1 matrix'
end
% return a horizontal vector
if cols == 1
    NAVps = NAVps';
end
% check the size of vector
[rows,cols] = size(BenchmarkReturns);
if min(rows,cols) ~= 1
    error 'BenchmarkReturns must be a 1xN or Nx1 matrix'
end
% return a horizontal vector
if cols == 1
    BenchmarkReturns = BenchmarkReturns';
end
N = length(BenchmarkReturns);
if length(NAVps) ~= N+1
    error 'length of NAVps must be 1 plus length of BenchmarkReturns'
end

% construct benchmark values
BenchmarkValues = convert_Returns_to_AssetValues(NAVps(1), BenchmarkReturns);

% relative values for each period
RelativeValues = NAVps - BenchmarkValues;

% calculate fund returns
FundReturn = (NAVps(2:end) - NAVps(1:end-1)) ./ NAVps(1:end-1);

% outperformance
Outperform = FundReturn - BenchmarkReturns;

% relative HWM
rHWM = calculate_HWM(RelativeValues);

% excess return
ExcessReturn = Outperform - HurdleRate;

if applyHWM     % apply rHWM condition
    % find period indices where relative value <= rHWM, i.e. rHWM condition not passed
    FailedPeriods = find((RelativeValues(2:end) <= rHWM(1:end-1)) == 1);
    FeeBeforeCF(FailedPeriods) = - HurdleRate .* NAVps(FailedPeriods) * FeeRate;
    
    PassedPeriods = find((RelativeValues(2:end) > rHWM(1:end-1)) == 1);
    % only increased relative value above rHWM count to effective overperformance
    Outperform_rHWM = Outperform(PassedPeriods) .* (RelativeValues(PassedPeriods+1) - rHWM(PassedPeriods)) ./ (RelativeValues(PassedPeriods+1) - RelativeValues(PassedPeriods));
    FeeBeforeCF(PassedPeriods) = (Outperform_rHWM - HurdleRate) .* NAVps(PassedPeriods) * FeeRate;

else           % don't apply rHWM condition
    FeeBeforeCF = ExcessReturn .* NAVps(1:end-1) * FeeRate;
end

% performance fee per share
PerfFeePs = max(min(FeeBeforeCF, CapPs), FloorPs);



